/**********************************************************************
* $Id$		lpc8xx_i2c.c		2011-06-02
*//**
* @file		lpc8xx_i2c.c
* @brief	Contains all functions support for I2C firmware library
* 			on lpc8xx
* @version	1.0
* @date		02. June. 2011
* @author	NXP MCU SW Application Team
*
* Copyright(C) 2011, NXP Semiconductor
* All rights reserved.
*
***********************************************************************
* Software that is described herein is for illustrative purposes only
* which provides customers with programming information regarding the
* products. This software is supplied "AS IS" without any warranties.
* NXP Semiconductors assumes no responsibility or liability for the
* use of the software, conveys no license or title under any patent,
* copyright, or mask work right to the product. NXP Semiconductors
* reserves the right to make changes in the software without
* notification. NXP Semiconductors also make no representation or
* warranty that such application will be suitable for the specified
* use without further testing or modification.
* Permission to use, copy, modify, and distribute this software and its
* documentation is hereby granted, under NXP Semiconductors
* relevant copyright in the software, without fee, provided that it
* is used in conjunction with NXP Semiconductors microcontrollers.  This
* copyright, permission, and disclaimer notice must appear in all copies of
* this code.
**********************************************************************/

/* Includes ------------------------------------------------------------------- */
#include "lpc8xx.h"
#include "lpc8xx_nmi.h"
#include "lpc8xx_i2c.h"

extern volatile uint8_t I2CSlaveTXBuffer[I2C_BUFSIZE];
extern volatile uint8_t I2CSlaveRXBuffer[I2C_BUFSIZE];

volatile uint32_t I2CStatus;

volatile uint32_t I2CSlvRXCount = 0;
volatile uint32_t I2CSlvTXCount = 0;
volatile uint32_t slvrxrdy = 0, slvtxrdy = 0;
volatile uint32_t slvaddrrcvd = 0;
volatile uint32_t SlaveAddr = 0;

/* Timeout related */
volatile uint32_t I2CSCLTimeoutCount = 0;
volatile uint32_t I2CEventTimeoutCount = 0;

/* Slave related */
volatile uint32_t I2CSlvSelectedCount = 0;
volatile uint32_t I2CSlvDeselectedCount = 0;
volatile uint32_t I2CSlvNotStrCount = 0;


#if TIMEOUT_ENABLED
/*****************************************************************************
** Function name:		I2C_I2CTimeoutStatus
**
** Descriptions:		I2C interrupt handler to handle timeout status
**
** parameters:			None
** Returned value:		None
** 
*****************************************************************************/
void I2C_I2CTimeoutStatus( LPC_I2C_TypeDef *I2Cx, uint32_t active )
{
  if(active & STAT_SCLTIMEOUT) {
		I2CSCLTimeoutCount++;
		I2Cx->STAT = STAT_SCLTIMEOUT;
  }
  if(active & STAT_EVTIMEOUT) {
		I2CEventTimeoutCount++;
		I2Cx->STAT = STAT_EVTIMEOUT;
  }
}
#endif


/*****************************************************************************
** Function name:		I2C_SlaveStatus
**
** Descriptions:		I2C interrupt handler to handle slave status
**
** parameters:			None
** Returned value:		None
** 
*****************************************************************************/
void I2C_SlaveStatus( LPC_I2C_TypeDef *I2Cx, uint32_t active )
{
  if (active & STAT_SLVNOTSTR) {
		I2CSlvNotStrCount++;
		I2Cx->INTENCLR = STAT_SLVNOTSTR;
  }
  if (active & STAT_SLVDESEL) {
		I2CSlvDeselectedCount++;
		I2Cx->STAT = STAT_SLVDESEL;
  }
  return;
}

/*****************************************************************************
** Function name:		I2C_IRQHandler
**
** Descriptions:		I2C interrupt handler
**
** parameters:			None
** Returned value:		None
** 
*****************************************************************************/
void I2C_IRQHandler(void)
{
  uint32_t active = LPC_I2C->INTSTAT;
  uint32_t i2cslvstate = LPC_I2C->STAT & SLAVE_STATE_MASK;
  uint32_t i;

     
  /* slave related interrupts. */
  I2C_SlaveStatus(LPC_I2C, active);

  if ( active & STAT_SLVPEND )
  {  	
		LPC_I2C->INTENCLR = STAT_SLVPEND;
		/* Below three states are for Slave mode: Address Received, TX, and RX. */
		switch ( i2cslvstate )
		{
			case STAT_SLVRX:
				slvrxrdy = 1;
				I2CSlaveRXBuffer[I2CSlvRXCount++] = LPC_I2C->SLVDAT;
				if ( I2CSlvRXCount != I2C_BUFSIZE )
				{
					LPC_I2C->SLVCTL = CTL_SLVCONTINUE;
				}
				else
				{
					LPC_I2C->SLVCTL = CTL_SLVNACK | CTL_SLVCONTINUE;
					for ( i = 0; i < I2C_BUFSIZE; i++ )
					{
						/* copy data from SlaveReceive buffer and send the received data back 
						to the master if "SlaveTX" is seen. */
						I2CSlaveTXBuffer[i] = I2CSlaveRXBuffer[i];
					}
				}
				LPC_I2C->INTENSET = STAT_SLVPEND; 
			break;
	  
			case STAT_SLVTX:
				slvtxrdy = 1;
				LPC_I2C->SLVDAT = I2CSlaveTXBuffer[I2CSlvTXCount++];
				LPC_I2C->SLVCTL = CTL_SLVCONTINUE;
				if ( I2CSlvTXCount == I2C_BUFSIZE )
				{
					I2CSlvTXCount = 0;
					for ( i = 0; i < I2C_BUFSIZE; i++ )	/* clear buffer */
					{
						/* If "SlaveTX" is done, clear the slave tx buffer. If the next packet is slave rx, 
						the slave tx will be filled again. If the next packet is slave tx again, "0"s will
						be sent out on the bus. */
						I2CSlaveTXBuffer[i] = 0;
					}
				} 
				LPC_I2C->INTENSET = STAT_SLVPEND;
			break;
	  
			case STAT_SLVADDR:
				/* slave address is received. */
				/* slaveaddrrcvd, slavetxrdy, and slaverxrdy are not used anymore
				as slave tx and rx are handled inside the ISR now. 
				I2C_SlaveSendData(), I2C_SlaveReceiveData(), I2C_SlaveReceiveSend()
				are for polling mode only. */
				slvaddrrcvd = 1;
				LPC_I2C->SLVCTL = CTL_SLVCONTINUE;
				LPC_I2C->INTENSET = STAT_SLVPEND; 
			break;
	  
			default:
			break;
		}
  }
  return;
}


/*****************************************************************************
** Function name:		I2C_SlvInit
**
** Descriptions:		I2C Slave mode initialization routine
**				
** parameters:			None
** Returned value:		None
** 
*****************************************************************************/
void I2C_SlvInit( LPC_I2C_TypeDef *I2Cx, uint32_t addr, uint32_t cfg, uint32_t clkdiv )
{
  I2Cx->CFG &= ~(CFG_SLVENA);
  slvtxrdy = slvrxrdy = 0;
  slvaddrrcvd = 0;

  I2Cx->DIV = clkdiv;

  /* Enable all addresses */
  I2Cx->SLVADR0 = addr;
  I2Cx->SLVADR1 = addr + 0x20;
  I2Cx->SLVADR2 = addr + 0x40; 
  I2Cx->SLVADR3 = addr + 0x60;

#if ADDR_QUAL_ENABLE
  /* RANGE (bit0 = 1) or MASK(bit0 = 0) mode */
#if 1
  /* RANGE (bit0 = 1) mode, (SLVADR0 <= addr <= SLVQUAL0) */
  I2Cx->SLVQUAL0 = ((LPC_I2C->SLVADR0 + 0x60) | 0x01);
#else
  /* MASK (bit0 = 0) mode, (addr & ~SLVQUAL0) = SLVADR0 */
  I2Cx->SLVQUAL0 = 0x06;
//  I2Cx->SLVQUAL0 = 0xfe;
#endif
#endif
  
#if I2C_INTERRUPT
  I2Cx->INTENSET |= (STAT_SLVDESEL | STAT_SLVNOTSTR);
  NVIC_DisableIRQ(I2C_IRQn);
  NVIC_ClearPendingIRQ(I2C_IRQn);
  NVIC_EnableIRQ(I2C_IRQn);
#endif
  I2Cx->CFG |= cfg;
  return;
}

#if I2C_MONITOR_MODE
/*****************************************************************************
** Function name:		I2C_MonInit
**
** Descriptions:		I2C Monitor mode initialization routine
**				
** parameters:			None
** Returned value:		None
** 
*****************************************************************************/
void I2C_MonInit( LPC_I2C_TypeDef *I2Cx, uint32_t cfg )
{
  I2Cx->CFG &= ~CFG_MONENA;
#if I2C_INTERRUPT
  I2Cx->INTENSET = STAT_MONRDY | STAT_MONOVERRUN | STAT_MONIDLE;
  NVIC_DisableIRQ(I2C_IRQn);
  NVIC_ClearPendingIRQ(I2C_IRQn);
  NVIC_EnableIRQ(I2C_IRQn);
#endif
  I2Cx->CFG |= cfg;
}
#endif

#if TIMEOUT_ENABLED
/*****************************************************************************
** Function name:		I2C_TimeoutInit
**
** Descriptions:		I2C Timeout initialization routine
**				
** parameters:			None
** Returned value:		None
** 
*****************************************************************************/
void I2C_TimeoutInit( LPC_I2C_TypeDef *I2Cx, uint32_t timeout_value )
{
  uint32_t to_value;
  
  I2Cx->CFG &= ~CFG_TIMEOUTENA;
  to_value = I2Cx->TIMEOUT & 0x000F;
  to_value |= (timeout_value << 4);
  I2Cx->TIMEOUT = to_value;  
#if I2C_INTERRUPT
  I2Cx->INTENSET |= (STAT_EVTIMEOUT|STAT_SCLTIMEOUT); 
  
  NVIC_DisableIRQ(I2C_IRQn);
  NVIC_ClearPendingIRQ(I2C_IRQn);
  NVIC_EnableIRQ(I2C_IRQn);
#endif
  I2Cx->CFG |= CFG_TIMEOUTENA;
}
#endif


/*****************************************************************************
** Function name:		I2C_SlaveSend
**
** Descriptions:		Send a block of data to the I2C port, the 
**									first parameter is the buffer pointer, the 2nd 
**									parameter is the block length.
**
** parameters:			buffer pointer, and the block length
** Returned value:	None
** 
*****************************************************************************/
void I2C_SlaveSendData( LPC_I2C_TypeDef *I2Cx, uint8_t *tx, uint32_t Length )
{
  uint32_t i;

  I2Cx->SLVCTL = CTL_SLVCONTINUE;
#if I2C_INTERRUPT
  I2Cx->INTENSET = STAT_SLVPEND;
#endif

  for ( i = 0; i < Length; i++ ) 
  {
#if I2C_INTERRUPT
		while(!slvtxrdy);
		slvtxrdy = 0;
		I2Cx->SLVDAT = *tx++;
		I2Cx->SLVCTL = CTL_SLVCONTINUE;
		I2Cx->INTENSET = STAT_SLVPEND; 
#else
		/* Move only if TX is ready */
		while ( (I2Cx->STAT & (STAT_SLVPEND|SLAVE_STATE_MASK)) != (STAT_SLVPEND|STAT_SLVTX) );
		I2Cx->SLVDAT = *tx++;
		I2Cx->SLVCTL = CTL_SLVCONTINUE;
#endif
  }
  return; 
}

/*****************************************************************************
** Function name:		I2C_SlaveReceive
** Descriptions:		the module will receive a block of data from 
**						the I2C, the 2nd parameter is the block 
**						length.
** parameters:			buffer pointer, and block length
** Returned value:		None
** 
*****************************************************************************/
void I2C_SlaveReceiveData( LPC_I2C_TypeDef *I2Cx, uint8_t *rx, uint32_t Length )
{
  uint32_t i;

  I2Cx->SLVCTL = CTL_SLVCONTINUE;
#if I2C_INTERRUPT
  I2Cx->INTENSET = STAT_SLVPEND;
#endif

  for ( i = 0; i < Length; i++ )
  {
#if I2C_INTERRUPT
		while(!slvrxrdy);
		slvrxrdy = 0;
		*rx++ = I2Cx->SLVDAT;
		if ( i != Length-1 ) {
			I2Cx->SLVCTL = CTL_SLVCONTINUE;
		}
		else {
			// I2Cx->SLVCTL = CTL_SLVNACK | CTL_SLVCONTINUE;
			I2Cx->SLVCTL = CTL_SLVCONTINUE;
		}
		I2Cx->INTENSET = STAT_SLVPEND; 
#else	
		while ( (I2Cx->STAT & (STAT_SLVPEND|SLAVE_STATE_MASK)) != (STAT_SLVPEND|STAT_SLVRX) );
		*rx++ = I2Cx->SLVDAT;
		if ( i != Length-1 ) {
			I2Cx->SLVCTL = CTL_SLVCONTINUE;
		}
		else {
			// I2Cx->SLVCTL = CTL_SLVNACK | CTL_SLVCONTINUE; 
			I2Cx->SLVCTL = CTL_SLVCONTINUE; 
		}
#endif
  }
  return; 
}

/*****************************************************************************
** Function name:		I2C_SlaveReceiveSend
** Descriptions:		Based on bit 0 of slave address, this module will
**						send or receive block of data to/from the master.
**						The length of TX and RX are the same. Two buffers,
**						one for TX and one for RX, are used.
** parameters:			tx buffer ptr, rx buffer ptr, and block length
** Returned value:		None
** 
*****************************************************************************/
void I2C_SlaveReceiveSend( LPC_I2C_TypeDef *I2Cx, uint8_t *tx, uint8_t *rx, uint32_t Length )
{
  uint32_t addr7bit;
  
    LPC_I2C->SLVCTL = CTL_SLVCONTINUE;
#if I2C_INTERRUPT
  I2Cx->INTENSET = STAT_SLVPEND;
  while(!slvaddrrcvd);
  slvaddrrcvd = 0;
#else
  while ( (I2Cx->STAT & (STAT_SLVPEND|SLAVE_STATE_MASK)) != (STAT_SLVPEND|STAT_SLVADDR) );
#endif
  SlaveAddr = I2Cx->SLVDAT;
  addr7bit = SlaveAddr & 0xFE;
#if ADDR_QUAL_ENABLE
  if ( I2Cx->SLVQUAL0 & 0xFF )
  {
		/* Either MASK or RANGE */
		if ( I2Cx->SLVQUAL0 & 0x01 ) {
			/* RANGE mode (SLVQUAL0 >= addr >= SLVADR0) */
			if ( (addr7bit < I2Cx->SLVADR0) || (addr7bit > (I2Cx->SLVQUAL0 & 0xFE)) ) {
				/* For debugging. That shouldn't happen. */
				while ( 1 );
			}
		}		
		else {
#if 0
			/* MASK mode */
			if ( (addr7bit & (I2Cx->SLVQUAL0&0xFE)) != I2Cx->SLVADR0 ) {
				/* For debugging. That shouldn't happen. */
				while ( 1 );
			}
#endif
		}		
  }
#else
  if ( ( addr7bit != I2Cx->SLVADR0 ) && ( addr7bit != I2Cx->SLVADR1 )
		&& ( addr7bit != I2Cx->SLVADR2 ) && ( addr7bit != I2Cx->SLVADR3 ) ) {
		/* For debugging. It should never happen, SLDADDR bit changes but addr doesn't match. */
		while ( 1 );
  }
#endif
   
  if ( (SlaveAddr & RD_BIT) == 0x00 ) {
		/* slave reads from master. */
		I2C_SlaveReceiveData( I2Cx, rx, Length ); 
  }
  else {
		/* slave write to master. */
		I2C_SlaveSendData( I2Cx, tx, Length ); 
  }
  return; 
}

//#endif /* _I2C */


/* --------------------------------- End Of File ------------------------------ */
